import React from "react";
import { useLanguage } from "@/contexts/LanguageContext";
import { SystemAnalysisData } from "@/types/analysis";
import InfoIcon from "@/components/InfoIcon";
import { CheckCircle, AlertTriangle, XCircle } from "lucide-react";

interface KPITabProps {
  systemData: SystemAnalysisData;
}

export default function KPITab({ systemData }: KPITabProps) {
  const { t } = useLanguage();
  const totalTickets = systemData.ticketStats.totalTickets;
  const completedTickets = systemData.ticketStats.completedTickets;
  const inProgressTickets = systemData.ticketStats.inProgressTickets;
  const openTickets = systemData.ticketStats.openTickets;

  const completionRate =
    totalTickets > 0 ? (completedTickets / totalTickets) * 100 : 0;
  const workInProgressRate =
    totalTickets > 0 ? (inProgressTickets / totalTickets) * 100 : 0;
  const pendingRate = totalTickets > 0 ? (openTickets / totalTickets) * 100 : 0;

  const avgResolutionTime = systemData.ticketStats.averageResolutionTime;

  const avgTicketsPerTechnician =
    systemData.technicianStats.length > 0
      ? systemData.technicianStats.reduce(
          (sum, tech) => sum + tech.totalTickets,
          0
        ) / systemData.technicianStats.length
      : 0;

  const topPerformer =
    systemData.technicianStats.length > 0
      ? systemData.technicianStats.reduce((top, tech) =>
          tech.completionRate > top.completionRate ? tech : top
        )
      : null;

  return (
    <>
      <div className='grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8'>
        <div className='bg-white p-6 rounded-lg shadow border-l-4 border-green-500'>
          <h3 className='text-sm font-medium text-gray-500 flex items-center'>
            {t("analysis.completionRate")}
            <InfoIcon
              tooltip={
                t("analysis.completionRateTooltip") ||
                "Percentage of tickets that have been completed out of all tickets. Higher is better - indicates good ticket closure rate."
              }
            />
          </h3>
          <p className='text-3xl font-bold text-green-600'>
            {completionRate.toFixed(1)}%
          </p>
          <p className='text-xs text-gray-500 mt-1'>
            {t("analysis.ticketsCompleted") || "Tickets completed vs total"}
          </p>
        </div>

        <div className='bg-white p-6 rounded-lg shadow border-l-4 border-blue-500'>
          <h3 className='text-sm font-medium text-gray-500 flex items-center'>
            {t("analysis.avgResolutionTime")}
            <InfoIcon
              tooltip={
                t("analysis.avgResolutionTimeTooltip") ||
                "Average time taken to resolve tickets from start to completion. Lower is better - indicates faster problem resolution."
              }
            />
          </h3>
          <p className='text-3xl font-bold text-blue-600'>
            {avgResolutionTime.toFixed(1)}h
          </p>
          <p className='text-xs text-gray-500 mt-1'>
            {t("analysis.timeFromStart")}
          </p>
        </div>

        <div className='bg-white p-6 rounded-lg shadow border-l-4 border-purple-500'>
          <h3 className='text-sm font-medium text-gray-500 flex items-center'>
            {t("analysis.workInProgress")}
            <InfoIcon
              tooltip={
                t("analysis.workInProgressTooltip") ||
                "Percentage of tickets currently being worked on by technicians. Moderate levels indicate good workflow - not too many idle tickets."
              }
            />
          </h3>
          <p className='text-3xl font-bold text-purple-600'>
            {workInProgressRate.toFixed(1)}%
          </p>
          <p className='text-xs text-gray-500 mt-1'>
            {t("analysis.ticketsBeingWorkedOn") ||
              "Tickets currently being worked on"}
          </p>
        </div>

        <div className='bg-white p-6 rounded-lg shadow border-l-4 border-orange-500'>
          <h3 className='text-sm font-medium text-gray-500 flex items-center'>
            {t("analysis.pendingRate")}
            <InfoIcon
              tooltip={
                t("analysis.pendingRateTooltip") ||
                "Percentage of tickets waiting to be started. Lower is better - indicates quick response to new tickets and good resource allocation."
              }
            />
          </h3>
          <p className='text-3xl font-bold text-orange-600'>
            {pendingRate.toFixed(1)}%
          </p>
          <p className='text-xs text-gray-500 mt-1'>
            {t("analysis.ticketsWaiting") || "Tickets waiting to be started"}
          </p>
        </div>
      </div>

      <div className='grid grid-cols-1 lg:grid-cols-2 gap-6 mb-8'>
        <div className='bg-white p-6 rounded-lg shadow'>
          <h3 className='text-lg font-semibold mb-4 flex items-center'>
            {t("analysis.technicianEfficiency") || "Technician Efficiency"}
            <InfoIcon
              tooltip={
                t("analysis.technicianEfficiencyTooltip") ||
                "Metrics showing how efficiently technicians are working, including average workload distribution and top performer identification."
              }
            />
          </h3>
          <div className='space-y-4'>
            <div className='flex justify-between items-center'>
              <span className='text-sm text-gray-600'>
                {t("analysis.avgTicketsPerTechnician") ||
                  "Avg Tickets per Technician"}
              </span>
              <span className='text-lg font-semibold text-gray-900'>
                {avgTicketsPerTechnician.toFixed(1)}
              </span>
            </div>
            <div className='flex justify-between items-center'>
              <span className='text-sm text-gray-600'>
                {t("analysis.totalTechnicians") || "Total Technicians"}
              </span>
              <span className='text-lg font-semibold text-gray-900'>
                {systemData.technicianStats.length}
              </span>
            </div>
            {topPerformer && (
              <div className='flex justify-between items-center'>
                <span className='text-sm text-gray-600'>
                  {t("analysis.topPerformer") || "Top Performer"}
                </span>
                <div className='text-right'>
                  <div className='text-sm font-medium text-gray-900'>
                    {topPerformer.technicianName}
                  </div>
                  <div className='text-xs text-gray-500'>
                    {topPerformer.completionRate.toFixed(1)}%{" "}
                    {t("analysis.completionLower") || "completion"}
                  </div>
                </div>
              </div>
            )}
          </div>
        </div>

        <div className='bg-white p-6 rounded-lg shadow'>
          <h3 className='text-lg font-semibold mb-4 flex items-center'>
            {t("analysis.responseTimeAnalysis") || "Response Time Analysis"}
            <InfoIcon
              tooltip={
                t("analysis.responseTimeAnalysisTooltip") ||
                "Analysis of how quickly tickets are being resolved, including average resolution times and daily ticket volumes."
              }
            />
          </h3>
          <div className='space-y-4'>
            <div className='flex justify-between items-center'>
              <span className='text-sm text-gray-600'>
                {t("analysis.averageResolution") || "Average Resolution"}
              </span>
              <span className='text-lg font-semibold text-gray-900'>
                {avgResolutionTime.toFixed(1)}h
              </span>
            </div>
            <div className='flex justify-between items-center'>
              <span className='text-sm text-gray-600'>
                {t("analysis.ticketsToday") || "Tickets Today"}
              </span>
              <span className='text-lg font-semibold text-gray-900'>
                {systemData.timeStats.todayTickets}
              </span>
            </div>
            <div className='flex justify-between items-center'>
              <span className='text-sm text-gray-600'>
                {t("analysis.avgPerDay") || "Avg per Day"}
              </span>
              <span className='text-lg font-semibold text-gray-900'>
                {systemData.timeStats.avgTicketsPerDay}
              </span>
            </div>
          </div>
        </div>
      </div>

      <div className='bg-white rounded-lg shadow mb-8'>
        <div className='px-6 py-4 border-b border-gray-200'>
          <h2 className='text-lg font-semibold flex items-center'>
            {t("analysis.performanceTargets") || "Performance Targets & Status"}
            <InfoIcon
              tooltip={
                t("analysis.performanceTargetsTooltip") ||
                "Visual indicators showing whether key performance targets are being met. Green check = target met, Yellow warning = needs attention, Red X = below target."
              }
            />
          </h2>
        </div>
        <div className='p-6'>
          <div className='grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6'>
            <div className='text-center'>
              <div className='text-2xl font-bold text-gray-900 mb-2 flex justify-center'>
                {completionRate >= 80 ? (
                  <CheckCircle className='w-8 h-8 text-green-500' />
                ) : completionRate >= 60 ? (
                  <AlertTriangle className='w-8 h-8 text-yellow-500' />
                ) : (
                  <XCircle className='w-8 h-8 text-red-500' />
                )}
              </div>
              <h4 className='font-medium text-gray-900'>
                {t("analysis.completionRate")}
              </h4>
              <p className='text-sm text-gray-500'>
                {t("analysis.target") || "Target"}: ≥80%
              </p>
              <p className='text-lg font-semibold text-gray-900'>
                {completionRate.toFixed(1)}%
              </p>
            </div>

            <div className='text-center'>
              <div className='text-2xl font-bold text-gray-900 mb-2 flex justify-center'>
                {avgResolutionTime <= 24 ? (
                  <CheckCircle className='w-8 h-8 text-green-500' />
                ) : avgResolutionTime <= 48 ? (
                  <AlertTriangle className='w-8 h-8 text-yellow-500' />
                ) : (
                  <XCircle className='w-8 h-8 text-red-500' />
                )}
              </div>
              <h4 className='font-medium text-gray-900'>
                {t("analysis.resolutionTime") || "Resolution Time"}
              </h4>
              <p className='text-sm text-gray-500'>
                {t("analysis.target") || "Target"}: ≤24h
              </p>
              <p className='text-lg font-semibold text-gray-900'>
                {avgResolutionTime.toFixed(1)}h
              </p>
            </div>

            <div className='text-center'>
              <div className='text-2xl font-bold text-gray-900 mb-2 flex justify-center'>
                {pendingRate <= 20 ? (
                  <CheckCircle className='w-8 h-8 text-green-500' />
                ) : pendingRate <= 40 ? (
                  <AlertTriangle className='w-8 h-8 text-yellow-500' />
                ) : (
                  <XCircle className='w-8 h-8 text-red-500' />
                )}
              </div>
              <h4 className='font-medium text-gray-900'>
                {t("analysis.pendingRate")}
              </h4>
              <p className='text-sm text-gray-500'>
                {t("analysis.target") || "Target"}: ≤20%
              </p>
              <p className='text-lg font-semibold text-gray-900'>
                {pendingRate.toFixed(1)}%
              </p>
            </div>
          </div>
        </div>
      </div>

      <div className='bg-white rounded-lg shadow'>
        <div className='px-6 py-4 border-b border-gray-200'>
          <h2 className='text-lg font-semibold flex items-center'>
            {t("analysis.technicianPerformanceSummary") ||
              "Technician Performance Summary"}
            <InfoIcon
              tooltip={
                t("analysis.technicianPerformanceSummaryTooltip") ||
                "Detailed performance metrics for each technician including completion rates, resolution times, and overall performance scores calculated from multiple factors."
              }
            />
          </h2>
        </div>
        <div className='overflow-x-auto relative'>
          <div className='min-w-full'>
            <table className='min-w-full divide-y divide-gray-200'>
              <thead className='bg-gray-50'>
                <tr>
                  <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                    {t("admin.technicians")}
                  </th>
                  <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                    <div className='flex items-center'>
                      {t("analysis.completionRate")}
                      <InfoIcon
                        tooltip={
                          t("analysis.completionRateTooltip2") ||
                          "Percentage of tickets completed by this technician out of their total assigned tickets."
                        }
                      />
                    </div>
                  </th>
                  <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                    <div className='flex items-center'>
                      {t("analysis.avgResolutionTime")}
                      <InfoIcon
                        tooltip={
                          t("analysis.avgResolutionTimeTooltip2") ||
                          "Average time this technician takes to resolve tickets from start to completion."
                        }
                      />
                    </div>
                  </th>
                  <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                    {t("analysis.totalTickets")}
                  </th>
                  <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                    <div className='flex items-center'>
                      {t("analysis.performanceScore") || "Performance Score"}
                      <InfoIcon
                        tooltip={
                          t("analysis.performanceScoreTooltip") ||
                          "Overall performance score (0-100) calculated from completion rate (40%), resolution time (30%), and ticket volume (30%)."
                        }
                      />
                    </div>
                  </th>
                </tr>
              </thead>
              <tbody className='bg-white divide-y divide-gray-200'>
                {systemData.technicianStats.map((tech) => {
                  const performanceScore = Math.min(
                    100,
                    tech.completionRate * 0.4 +
                      (tech.avgResolutionTime <= 24
                        ? 100
                        : Math.max(
                            0,
                            100 - (tech.avgResolutionTime - 24) * 2
                          )) *
                        0.3 +
                      Math.min(100, tech.totalTickets * 5) * 0.3
                  );

                  return (
                    <tr key={tech.technicianId}>
                      <td className='px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900'>
                        {tech.technicianName}
                      </td>
                      <td className='px-6 py-4 whitespace-nowrap text-sm text-gray-500'>
                        <span
                          className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${
                            tech.completionRate >= 80
                              ? "bg-green-100 text-green-800"
                              : tech.completionRate >= 60
                              ? "bg-yellow-100 text-yellow-800"
                              : "bg-red-100 text-red-800"
                          }`}
                        >
                          {tech.completionRate.toFixed(1)}%
                        </span>
                      </td>
                      <td className='px-6 py-4 whitespace-nowrap text-sm text-gray-500'>
                        {tech.avgResolutionTime.toFixed(1)}h
                      </td>
                      <td className='px-6 py-4 whitespace-nowrap text-sm text-gray-500'>
                        {tech.totalTickets}
                      </td>
                      <td className='px-6 py-4 whitespace-nowrap text-sm text-gray-500'>
                        <div className='flex items-center'>
                          <div className='w-16 bg-gray-200 rounded-full h-2 mr-2'>
                            <div
                              className={`h-2 rounded-full ${
                                performanceScore >= 80
                                  ? "bg-green-500"
                                  : performanceScore >= 60
                                  ? "bg-yellow-500"
                                  : "bg-red-500"
                              }`}
                              style={{ width: `${performanceScore}%` }}
                            ></div>
                          </div>
                          <span className='text-sm font-medium'>
                            {performanceScore.toFixed(0)}
                          </span>
                        </div>
                      </td>
                    </tr>
                  );
                })}
              </tbody>
            </table>
          </div>
        </div>
      </div>
    </>
  );
}
